'use client';

import { useState, useRef, useEffect } from 'react';
import { useToast } from '@/components/ui/ToastContext';

interface ExperienceDocument {
  id: string;
  filename: string;
  fileUrl: string;
  fileSize: string;
  fileType: string;
}

interface Experience {
  id: string;
  title: string;
  company: string;
  location: string;
  startDate: string;
  endDate?: string;
  isCurrent: boolean;
  description: string;
  documents: ExperienceDocument[];
  createdAt: string;
  updatedAt: string;
}

interface AddExperienceModalProps {
  isOpen: boolean;
  onClose: () => void;
  experience?: Experience | null;
  onExperienceUpdated: () => void;
}

export default function AddExperienceModal({ 
  isOpen, 
  onClose, 
  experience, 
  onExperienceUpdated 
}: AddExperienceModalProps) {
  const [formData, setFormData] = useState({
    title: '',
    company: '',
    location: '',
    startDate: '',
    endDate: '',
    isCurrent: false,
    description: ''
  });
  const [selectedFiles, setSelectedFiles] = useState<File[]>([]);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [errors, setErrors] = useState<Record<string, string>>({});
  const fileInputRef = useRef<HTMLInputElement>(null);
  const { showError, showSuccess } = useToast();
  const [existingFiles, setExistingFiles] = useState<ExperienceDocument[]>([]);

  // Initialize form data when experience prop changes
  useEffect(() => {
    if (experience) {
      setFormData({
        title: experience.title || '',
        company: experience.company || '',
        location: experience.location || '',
        startDate: experience.startDate ? new Date(experience.startDate).toISOString().split('T')[0] : '',
        endDate: experience.endDate ? new Date(experience.endDate).toISOString().split('T')[0] : '',
        isCurrent: experience.isCurrent || false,
        description: experience.description || ''
      });
      if (experience.documents && Array.isArray(experience.documents)) {
      setExistingFiles(experience.documents);
    }
    } else {
      setFormData({
        title: '',
        company: '',
        location: '',
        startDate: '',
        endDate: '',
        isCurrent: false,
        description: ''
      });
    }
    setSelectedFiles([]);
    setErrors({});
  }, [experience, isOpen]);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value, type } = e.target;

    setFormData(prev => ({
      ...prev,
      [name]: type === 'checkbox' ? (e.target as HTMLInputElement).checked : value
    }));

    // Clear error when user starts typing
    if (errors[name]) {
      setErrors(prev => ({
        ...prev,
        [name]: ''
      }));
    }
  };

  const handleFileSelect = (files: FileList) => {
    const newFiles = Array.from(files);
    const validFiles: File[] = [];
    
    newFiles.forEach(file => {
      // Validate file type
      if (!file.type.match(/^(image|application\/pdf|text|application\/msword|application\/vnd\.openxmlformats-officedocument\.wordprocessingml\.document).*$/)) {
        showError(`File "${file.name}" is not a supported format`);
        return;
      }
      
      // Validate file size (max 10MB)
      if (file.size > 10 * 1024 * 1024) {
        showError(`File "${file.name}" exceeds 10MB limit`);
        return;
      }
      
      validFiles.push(file);
    });
    
    setSelectedFiles(prev => [...prev, ...validFiles]);
  };

  const handleDrop = (e: React.DragEvent<HTMLDivElement>) => {
    e.preventDefault();
    const files = e.dataTransfer.files;
    if (files.length > 0) {
      handleFileSelect(files);
    }
  };

  const handleDragOver = (e: React.DragEvent<HTMLDivElement>) => {
    e.preventDefault();
  };

  const handleFileInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = e.target.files;
    if (files && files.length > 0) {
      handleFileSelect(files);
    }
  };

  const handleBrowseClick = () => {
    fileInputRef.current?.click();
  };

  const handleRemoveFile = (index: number) => {
    setSelectedFiles(prev => prev.filter((_, i) => i !== index));
  };
const validateForm = () => {
  const newErrors: Record<string, string> = {};

  // --- Title ---
  if (!formData.title.trim()) {
    newErrors.title = "Title is required";
  } else if (formData.title.trim().length < 2) {
    newErrors.title = "Title must be at least 2 characters long";
  }

  // --- Company ---
  if (!formData.company.trim()) {
    newErrors.company = "Company name is required";
  } else if (formData.company.trim().length < 2) {
    newErrors.company = "Company name must be at least 2 characters long";
  }

  // --- Start Date ---
  if (!formData.startDate.trim()) {
    newErrors.startDate = "Start date is required";
  }

  // --- End Date ---
  if (!formData.isCurrent) {
    if (!formData.endDate.trim()) {
      newErrors.endDate = "End date is required when not current position";
    } else if (
      formData.startDate &&
      new Date(formData.startDate) > new Date(formData.endDate)
    ) {
      newErrors.endDate = "End date must be after start date";
  }
  }

  // --- Location ---
  if (!formData.location.trim()) {
    newErrors.location = "Location is required";
  } else if (formData.location.trim().length < 2) {
    newErrors.location = "Location must be at least 2 characters long";
  }

  // --- Description ---
  if (!formData.description.trim()) {
    newErrors.description = "Description is required";
  } else if (formData.description.trim().length < 10) {
    newErrors.description = "Description must be at least 10 characters long";
  }

  // --- Uploaded Files ---
    const hasExistingFiles = existingFiles && existingFiles.length > 0;
  const hasNewFiles = selectedFiles && selectedFiles.length > 0;

  if (!hasExistingFiles && !hasNewFiles) {
    newErrors.files = "Please upload at least one document or image";
  } else if (hasNewFiles) {
    // Validate new files only
    const invalidFile = selectedFiles.find((file) => {
      const validTypes = [
        "image/jpeg",
        "image/png",
        "application/pdf",
        "application/msword",
        "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
        "text/plain",
      ];
      const maxSize = 5 * 1024 * 1024; // 5MB limit
      return !validTypes.includes(file.type) || file.size > maxSize;
    });

    if (invalidFile) {
      newErrors.files =
        "One or more files have an invalid format or exceed 5MB";
    }
  }

  setErrors(newErrors);
  return Object.keys(newErrors).length === 0;
};

const handleSubmit = async (e: React.FormEvent) => {
  e.preventDefault();

  if (!validateForm()) return;

  setIsSubmitting(true);
  try {
    const submitData = new FormData();
    submitData.append('title', formData.title.trim());
    submitData.append('company', formData.company.trim());
    submitData.append('location', formData.location.trim());
    submitData.append('startDate', formData.startDate);
    if (formData.endDate) submitData.append('endDate', formData.endDate);
    submitData.append('isCurrent', formData.isCurrent.toString());
    submitData.append('description', formData.description.trim());

    // ✅ Append new files
    selectedFiles.forEach(file => {
      submitData.append('documents', file);
    });

    // ✅ Append remaining existing file IDs (if applicable)
    if (experience && existingFiles.length > 0) {
      const existingFileIds = existingFiles.map(file => file.id);
      submitData.append('existingFileIds', JSON.stringify(existingFileIds));
    }

    const url = experience 
      ? `/api/freelancer/experiences/${experience.id}`
      : '/api/freelancer/experiences';

    const method = experience ? 'PUT' : 'POST';

    const response = await fetch(url, { method, body: submitData });
    const data = await response.json();

    if (!response.ok || !data.success) {
      throw new Error(data.error || 'Failed to save experience');
    }

    showSuccess(experience ? 'Experience updated successfully!' : 'Experience added successfully!');
    onExperienceUpdated();
  } catch (error: any) {
    console.error('Error submitting experience:', error);
    showError(error.message || 'Failed to save experience');
  } finally {
    setIsSubmitting(false);
  }
};


  const handleClose = () => {
    if (!isSubmitting) {
      setFormData({
        title: '',
        company: '',
        location: '',
        startDate: '',
        endDate: '',
        isCurrent: false,
        description: ''
      });
      setSelectedFiles([]);
      setErrors({});
      onClose();
    }
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };
const handleRemoveExistingFile = (index: number) => {
  const updated = existingFiles.filter((_, i) => i !== index);
  setExistingFiles(updated);
};

  if (!isOpen) return null;

  return (
    <div className="modal fade show custom-modal modal-overlay add-experience-modal" onClick={handleClose}>
      <div className='modal-dialog w-100 modal-dialog-centered'>
        <div className="modal-content" onClick={(e) => e.stopPropagation()}>
          <div className="modal-header-borderless tab-title mb-4">
            <h5>{experience ? 'Edit Experience' : 'Add Experience'}</h5>
            <button 
              className="btn-close" 
              onClick={handleClose}
              disabled={isSubmitting}
            >
            </button>
          </div>

          <div className="modal-body-no-pb">
            <form onSubmit={handleSubmit}>
              <div className="form-group">
                <div className="mb-20">
                  <label htmlFor="title" className="form-label">Title</label>
                  <div className="input-icon">
                    <input
                      type="text"
                      className={`form-control ${errors.title ? 'error' : ''}`}
                      name="title"
                      id="title"
                      placeholder="Enter Job Title"
                      value={formData.title}
                      onChange={handleInputChange}
                      disabled={isSubmitting}
                    />
                  </div>
                  {errors.title && <div className="error-message">{errors.title}</div>}
                </div>

                <div className="mb-20">
                  <label htmlFor="company" className="form-label">Company</label>
                  <div className="input-icon">
                    <input
                      type="text"
                      className={`form-control ${errors.company ? 'error' : ''}`}
                      name="company"
                      id="company"
                      placeholder="Enter Company Name"
                      value={formData.company}
                      onChange={handleInputChange}
                      disabled={isSubmitting}
                    />
                  </div>
                  {errors.company && <div className="error-message">{errors.company}</div>}
                </div>
              </div>

              <div className="form-group">
                <div className="mb-20">
                  <label htmlFor="startDate" className="form-label">Start Date</label>
                  <div className="input-icon">
                    <input
                      type="date"
                      className={`form-control ${errors.startDate ? 'error' : ''}`}
                      name="startDate"
                      id="startDate"
                      value={formData.startDate}
                      onChange={handleInputChange}
                      disabled={isSubmitting}
                    />
                  </div>
                  {errors.startDate && <div className="error-message">{errors.startDate}</div>}
                </div>

                <div className="mb-20">
                  <label htmlFor="endDate" className="form-label">End Date</label>
                  <div className="input-icon">
                    <input
                      type="date"
                      className={`form-control ${errors.endDate ? 'error' : ''}`}
                      name="endDate"
                      id="endDate"
                      value={formData.endDate}
                      onChange={handleInputChange}
                      disabled={isSubmitting || formData.isCurrent}
                    />
                  </div>
                  {errors.endDate && <div className="error-message">{errors.endDate}</div>}
                </div>
              </div>

              <div className="form-group single-column">
                <div className="mb-20">
                  <div className="form-check">
                    <input
                      type="checkbox"
                      className="form-check-input"
                      name="isCurrent"
                      id="isCurrent"
                      checked={formData.isCurrent}
                      onChange={handleInputChange}
                      disabled={isSubmitting}
                    />
                    <label htmlFor="isCurrent" className="form-check-label">
                      I currently work here
                    </label>
                  </div>
                </div>
              </div>

              <div className="form-group single-column">
                <div className="mb-20">
                  <label htmlFor="location" className="form-label">Location</label>
                  <div className="input-icon">
                    <input
                      type="text"
                      className={`form-control ${errors.location ? 'error' : ''}`}
                      name="location"
                      id="location"
                      placeholder="Enter Location"
                      value={formData.location}
                      onChange={handleInputChange}
                      disabled={isSubmitting}
                    />
                  </div>
                  {errors.location && <div className="error-message">{errors.location}</div>}
                </div>
              </div>

          <div className="upload-outer mb-20">
  <label className="form-label fw-semibold">Upload Documents</label>
  <div 
    className="upload-area"
    onDrop={handleDrop}
    onDragOver={handleDragOver}
  >
    {(existingFiles.length > 0 || selectedFiles.length > 0) ? (
      <div className="files-preview">
        {/* Show existing uploaded files */}
        {existingFiles.map((file, index) => (
          <div key={`existing-${index}`} className="file-preview-item">
            <div className="file-info">
              <a 
                href={file.fileUrl} 
                target="_blank" 
                rel="noopener noreferrer" 
                className="file-name"
              >
                {file.filename}
              </a>
              <div className="file-size">{file.fileSize}</div>
            </div>
            <button
              type="button"
              className="remove-file-btn"
              onClick={() => handleRemoveExistingFile(index)}
              disabled={isSubmitting}
            >
              &times;
            </button>
          </div>
        ))}

        {/* Show new files not yet uploaded */}
        {selectedFiles.map((file, index) => (
          <div key={`new-${index}`} className="file-preview-item">
            <div className="file-info">
              <div className="file-name">{file.name}</div>
              <div className="file-size">{formatFileSize(file.size)}</div>
            </div>
            <button 
              type="button" 
              className="remove-file-btn"
              onClick={() => handleRemoveFile(index)}
              disabled={isSubmitting}
            >
              &times;
            </button>
          </div>
        ))}

        <div className="add-more-files">
          <button 
            type="button" 
            className="browse-btn" 
            onClick={handleBrowseClick}
            disabled={isSubmitting}
          >
            Add more files
          </button>
        </div>
      </div>
    ) : (
      <div className="upload-placeholder">
        {/* ... (your SVG placeholder remains same) ... */}
        <div className="upload-img-content">
          <p>
            Drag & drop files or{' '}
            <button 
              type="button" 
              className="browse-btn" 
              onClick={handleBrowseClick}
              disabled={isSubmitting}
            >
              Browse
            </button>
          </p>
          <small className="text-muted">Supported formats: Images, PDF, Word documents</small>
        </div>
      </div>
    )}
    
    <input 
      type="file" 
      ref={fileInputRef}
      onChange={handleFileInputChange}
      accept="image/*,application/pdf,.doc,.docx,.txt"
      className="file-input"
      multiple
      disabled={isSubmitting}
    />
  </div>

  {errors.files && (
    <div className="error-message">{errors.files}</div>
  )}
</div>


              <div className="form-group-box">
                <div className="mb-20">
                  <label htmlFor="description" className="form-label">Description</label>
                  <div className="input-icon">
                    <textarea 
                      className="form-control" 
                      name="description"
                      id="description"
                      rows={5} 
                      placeholder="Describe your experience..."
                      value={formData.description}
                      onChange={handleInputChange}
                      disabled={isSubmitting}
                    />
                  </div>
                </div>
                 {errors.description && <div className="error-message">{errors.description}</div>}
              </div>

              <div className="modal-actions">
                <button 
                  type="button" 
                  className="custom-btn secondary-outline" 
                  onClick={handleClose}
                  disabled={isSubmitting}
                >
                  Cancel
                </button>
                
                <button 
                  type="submit" 
                  className="custom-btn primary-fill"
                  disabled={isSubmitting}
                >
                  {isSubmitting ? 'Saving...' : (experience ? 'Update' : 'Save')}
                </button>
              </div>
            </form>
          </div>
        </div>
      </div>
      
      <style jsx>{`
        .modal-overlay {
          position: fixed;
          top: 0;
          left: 0;
          right: 0;
          bottom: 0;
          background: rgba(0, 0, 0, 0.7);
          display: flex;
          align-items: center;
          justify-content: center;
          z-index: 9999;
          padding: 20px;
        }

        .modal-content {
          background: var(--color-white);
          border-radius: 15px;
          width: 100%;
          max-height: 90vh;
          overflow-y: auto;
          box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
          animation: modalFadeIn 0.3s ease-out;
        }

        @keyframes modalFadeIn {
          from {
            opacity: 0;
            transform: scale(0.9) translateY(-20px);
          }
          to {
            opacity: 1;
            transform: scale(1) translateY(0);
          }
        }

        .modal-header {
          display: flex;
          justify-content: space-between;
          align-items: center;
          padding: 25px 30px 20px;
          border-bottom: 1px solid var(--border-color);
        }

        .modal-header h5 {
          font-size: var(--font-18);
          font-weight: 700;
          color: var(--typo-secondary);
          margin: 0;
        }

        .modal-close {
          background: none;
          border: none;
          padding: 8px;
          cursor: pointer;
          color: var(--text-color);
          border-radius: 6px;
          transition: all 0.3s ease;
        }

        .modal-close:hover:not(:disabled) {
          background: var(--light-orange);
          color: var(--color-primary);
        }

        .modal-close:disabled {
          opacity: 0.5;
          cursor: not-allowed;
        }

        .modal-body {
          padding: 25px 30px;
        }

        .form-group {
          display: grid;
          grid-template-columns: 1fr 1fr;
          gap: 20px;
          margin-bottom: 20px;
        }

        .form-group.single-column {
          grid-template-columns: 1fr;
        }

        .mb-20 {
          margin-bottom: 15px;
        }

        .form-label {
          display: block;
          margin-bottom: 8px;
          font-weight: 600;
          font-size: var(--font-small);
          color: var(--typo-secondary);
        }

        .form-label:after {
          content: " *";
          color: #e74c3c;
        }

        .input-icon {
          position: relative;
          width: 100%;
        }

        .form-control {
          width: 100%;
          padding: 12px 15px;
          border: 1px solid var(--border-color);
          border-radius: 8px;
          font-size: var(--font-small);
          color: var(--typo-secondary);
          background: var(--color-white);
          transition: all 0.3s ease;
        }

        .form-control:focus {
          outline: none;
          border-color: var(--color-primary);
          box-shadow: 0 0 0 2px rgba(239, 115, 16, 0.1);
        }

        .form-control.error {
          border-color: #e74c3c;
        }

        .form-control::placeholder {
          color: var(--text-color);
        }

        .error-message {
          color: #e74c3c;
          font-size: var(--font-xs);
          margin-top: 5px;
        }

        .upload-outer {
          margin-bottom: 20px;
        }

        .form-label.fw-semibold {
          font-weight: 600;
        }

        .upload-area {
          position: relative;
          border: 2px dashed var(--border-color);
          border-radius: 12px;
          padding: 40px 20px;
          text-align: center;
          transition: all 0.3s ease;
          cursor: pointer;
          min-height: 200px;
          display: flex;
          align-items: center;
          justify-content: center;
        }

        .upload-area:hover {
          border-color: var(--color-primary);
          background: rgba(239, 115, 16, 0.02);
        }

        .upload-placeholder {
          display: flex;
          flex-direction: column;
          align-items: center;
          gap: 15px;
        }

        .upload-placeholder p {
          font-size: var(--font-small);
          color: var(--typo-secondary);
          margin: 0;
        }

        .browse-btn {
          background: none;
          border: none;
          color: var(--color-primary);
          text-decoration: underline;
          cursor: pointer;
          font-size: inherit;
          padding: 0;
        }

        .browse-btn:hover:not(:disabled) {
          opacity: 0.8;
        }

        .browse-btn:disabled {
          opacity: 0.5;
          cursor: not-allowed;
        }

        .text-muted {
          color: var(--text-color);
          font-size: var(--font-xs);
        }

        .file-input {
          display: none;
        }

        .files-preview {
          width: 100%;
        }

        .file-preview-item {
          display: flex;
          justify-content: space-between;
          align-items: center;
          padding: 10px 15px;
          background: var(--bg-light);
          border-radius: 8px;
          margin-bottom: 10px;
        }

        .file-info {
          flex: 1;
        }

        .file-name {
          font-size: var(--font-small);
          font-weight: 600;
          color: var(--typo-secondary);
          margin-bottom: 2px;
        }

        .file-size {
          font-size: var(--font-xs);
          color: var(--text-color);
        }

        .remove-file-btn {
          background: #e74c3c;
          color: white;
          border: none;
          border-radius: 50%;
          width: 24px;
          height: 24px;
          cursor: pointer;
          display: flex;
          align-items: center;
          justify-content: center;
          font-size: 16px;
          font-weight: bold;
          transition: all 0.3s ease;
        }

        .remove-file-btn:hover:not(:disabled) {
          background: #c0392b;
          transform: scale(1.1);
        }

        .remove-file-btn:disabled {
          opacity: 0.5;
          cursor: not-allowed;
        }

        .add-more-files {
          text-align: center;
          padding: 15px;
          border: 1px dashed var(--border-color);
          border-radius: 8px;
          margin-top: 10px;
        }

        .modal-actions {
          display: flex;
          justify-content: flex-end;
          gap: 15px;
          margin-top: 25px;
          padding-top: 20px;
          border-top: 1px solid var(--border-color);
        }

        .custom-btn {
          display: inline-flex;
          align-items: center;
          justify-content: center;
          padding: 10px 20px;
          border-radius: 8px;
          font-size: var(--font-small);
          font-weight: 600;
          text-decoration: none;
          cursor: pointer;
          transition: all 0.3s ease;
          border: 2px solid transparent;
          min-width: 100px;
        }

        .custom-btn.primary-fill {
          background: var(--color-primary);
          color: var(--color-white);
          border-color: var(--color-primary);
        }

        .custom-btn.primary-fill:hover:not(:disabled) {
          background: transparent;
          color: var(--color-primary);
        }

        .custom-btn.secondary-outline {
          background: transparent;
          color: var(--text-color);
          border-color: var(--border-color);
        }

        .custom-btn.secondary-outline:hover:not(:disabled) {
          background: var(--border-color);
          color: var(--typo-secondary);
        }

        .custom-btn:disabled {
          opacity: 0.6;
          cursor: not-allowed;
        }

        @media (max-width: 767px) {
          .modal-content {
            margin: 10px;
            max-height: calc(100vh - 20px);
          }

          .modal-header {
            padding: 20px;
          }

          .modal-body {
            padding: 20px;
          }

          .form-group {
            grid-template-columns: 1fr;
            gap: 0;
          }

          .upload-area {
            padding: 30px 15px;
          }

          .modal-actions {
            flex-direction: column;
            gap: 10px;
          }

          .custom-btn {
            width: 100%;
          }
        }
      `}</style>
    </div>
  );
}