'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { BidFormData, BidModalProps } from '@/types/bid';
import { getCountryDataByDomain } from '@/lib/countries';

export default function BidModal({
  isOpen,
  onClose,
  onSubmit,
  projectTitle,
  loading = false,
  isUpdateMode = false,
  initialBidData,
  projectBudget,
  projectDuration,
  projectProposal
}: BidModalProps) {

  console.log(projectProposal,'zzzzzzzzzzzzzzzzzzzzzzzzzzz');

  const [bidForm, setBidForm] = useState<BidFormData>({
    price: initialBidData?.amount || '',
    days: initialBidData?.deliveryTime || '',
    description: initialBidData?.proposal || ''
  });

  const [isSubmitting, setIsSubmitting] = useState(false);
  const datacounter = getCountryDataByDomain();

  // Update form when initialBidData changes (for update mode)
  useEffect(() => {
    if (isOpen && initialBidData) {
      setBidForm({
        price: initialBidData.amount || '',
        days: initialBidData.deliveryTime || '',
        description: initialBidData.proposal || ''
      });
    } else if (isOpen && !initialBidData) {
      // Reset form for new bids
      setBidForm({ price: '', days: '', description: '' });
    }
  }, [isOpen, initialBidData]);

  const [validationError, setValidationError] = useState<{
    description: string
  } | null>(null);

  const [showInfo, setShowInfo] = useState<{
    budget: string | null,
    duration: string | null
  } | null>(null);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!bidForm.price || !bidForm.days || !bidForm.description) {
      return;
    }

    const errorObj: {
      description: string
    } = {
      description: ""
    }

    if ((bidForm.description.trim()).length < 10) {
      errorObj.description = "Description cannot have length less than 10 characters";
    }


    if (errorObj.description) {
      setValidationError(errorObj);
      return;
    }

    setIsSubmitting(true);

    try {
      await onSubmit(bidForm);
      // Reset form after successful submission
      setBidForm({ price: '', days: '', description: '' });
      onClose();

      setTimeout(() => {
        window.location.reload();   // true forces reload from server; default is fine for most cases
      }, 1500)


    } catch (error) {
      console.error('Error submitting bid:', error);
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleClose = () => {
    if (!isSubmitting) {
      setBidForm({ price: '', days: '', description: '' });
      setValidationError(null);
      setShowInfo(null);
      onClose();
    }
  };

  const handleInputChange = (field: keyof BidFormData, value: string) => {
    setBidForm(prev => ({
      ...prev,
      [field]: value
    }));
    setShowInfo((prev: any) => {
      let updated = { ...prev };

      if (field === "price") {
        if (projectBudget && Number(value) > projectBudget) {
          updated.budget = "You have entered budget that is greater than the project's budget";
        } else {
          updated.budget = null;
        }
      }

      if (field === "days") {
        if (projectDuration && Number(value) > projectDuration) {
          updated.duration = "You have entered duration that is greater than the project's duration";
        } else {
          updated.duration = null;
        }
      }

      return updated;
    });
  };

  if (!isOpen) return null;

  return (
    <>
      <div className="modal fade show custom-modal" style={{ display: 'block' }} tabIndex={-1}>
        <div className="modal-dialog modal-dialog-centered">
          <div className="modal-content">
            <div className="tab-title">
              <h5>{isUpdateMode ? 'Update Proposal' : 'Add Proposal'}</h5>
              <button
                type="button"
                className="btn-close"
                onClick={handleClose}
                disabled={isSubmitting}
                aria-label="Close"
              />
            </div>

            {projectTitle && (
              <div className="mb-3">
                <small className="text-muted nodata-font">
                  Bidding on: <strong>{projectTitle}</strong>
                </small> {" "}
                {projectBudget && <small className="text-muted nodata-font">
                  Price: <strong>{datacounter?.currencySymbol ?? ''} {projectBudget}</strong>
                </small>}{" "}
                {projectDuration && <small className="text-muted nodata-font">
                  Work Duration: <strong>{projectDuration}</strong> {projectDuration> 9 ? 'Days':'Days'}
                </small>}
              </div>
            )}

            <form onSubmit={handleSubmit}>
              <div className="form-group">
                <div className="mb-20">
                  
                  <label htmlFor="bid-price" className="form-label">
                    Price * <span className="text-muted">(in {datacounter?.currencySymbol ?? ''})</span>
                  </label>
                  <div className="input-icon">
                    <input
                      type="number"
                      className="form-control"
                      name="price"
                      id="bid-price"
                      placeholder="Enter your bid amount"
                      value={bidForm.price}
                      onChange={(e) => handleInputChange('price', e.target.value)}
                      min="1"
                      step="0.01"
                      required
                      disabled={isSubmitting}
                    />
                  </div>
                  {showInfo && showInfo.budget && <span  style={{ color: "#069ae7" }}>{showInfo.budget}.</span>}
                </div>

                <div className="mb-20">                  
                  <label htmlFor="bid-days" className="form-label">
                    Days needed to complete the work *
                  </label>
                  <div className="input-icon">
                    <input
                      type="number"
                      className="form-control"
                      name="days"
                      id="bid-days"
                      placeholder="Enter number of days"
                      value={bidForm.days}
                      onChange={(e) => handleInputChange('days', e.target.value)}
                      min="1"
                      required
                      disabled={isSubmitting}
                    />
                  </div>
                  {showInfo && showInfo.duration && <span  style={{ color: "#069ae7" }}>{showInfo.duration}.</span>}
                </div>
              </div>

              <div className="form-group">
                <div className="mb-20">
                  {validationError && <span>{validationError.description}</span>}
                  <label htmlFor="bid-description" className="form-label">
                    Proposal Description *
                  </label>
                  <div className="input-icon">
                    <textarea
                      className="form-control"
                      name="description"
                      id="bid-description"
                      placeholder="Describe your approach, experience, and why you're the best fit for this project..."
                      value={bidForm.description}
                      onChange={(e) => handleInputChange('description', e.target.value)}
                      required
                      disabled={isSubmitting}
                      rows={6}
                    />
                  </div>
                </div>
              </div>

              <div className="pop-dis">
                <p>
                  Please read in full and understand the client's requirements. A well written and fully understood proposal is key to winning the work. See our{' '}
                  <Link href="/proposal-advice">Proposal Advice Here.</Link>
                </p>
              </div>

              <div className="d-flex gap-2">
                <button
                  type="button"
                  className="flex-fill custom-btn outline-btn cancel-modal-btn"
                  onClick={handleClose}
                  disabled={isSubmitting}
                >
                  Cancel
                </button>
                <button
                  type="submit"
                  className="flex-fill custom-btn secondary-btn"
                  // disabled={isSubmitting || !bidForm.price || !bidForm.days || !bidForm.description}
                  disabled={isSubmitting}
                >
                  {isSubmitting ? (
                    <>
                      <span
                        className="spinner-border spinner-border-sm me-2"
                        role="status"
                        aria-hidden="true"
                      />
                      Submitting...
                    </>
                  ) : (
                    isUpdateMode ? 'Update Proposal' : 'Submit Proposal'
                  )}
                </button>
              </div>
            </form>
          </div>
        </div>
      </div>

      {/* Modal Backdrop */}
      <div
        className="modal-backdrop fade show"
        onClick={handleClose}
        style={{ pointerEvents: isSubmitting ? 'none' : 'auto' }}
      />
    </>
  );
}