"use client";

import { useState, useEffect } from "react";
import ConversationListNew from "./ConversationListNew";
import "./messages.css";
import MessageThread from "./MessageThread";

interface Profile {
  firstName?: string;
  lastName?: string;
  profileImage?: string;
}

interface User {
  id: string;
  email: string;
  profile?: Profile;
}

interface MessagesInterfaceProps {
  projectId?: string;
  contactId?: string;
  preselectedContactId?: string;
  projectStatus?: string;
  conversationId?: string;
}

export default function MessagesInterface({
  projectId,
  contactId,
  preselectedContactId,
  projectStatus,
  conversationId: propConversationId,
}: MessagesInterfaceProps) {
  const [selectedContact, setSelectedContact] = useState<User | null>(null);
  const [selectedContactId, setSelectedContactId] = useState<string | undefined>(contactId);
  const [conversationId, setConversationId] = useState<string>(propConversationId || ""); // Initialize with prop
  const [shouldSelectFirst, setShouldSelectFirst] = useState<boolean>(!contactId && !preselectedContactId && !propConversationId);
  const [selectedProjectId, setSelectedProjectId] = useState<string>(projectId || "");
  const [selectedProjectStatus, setSelectedProjectStatus] = useState<string>(projectStatus || "");
  const [showSidebar, setShowSidebar] = useState<boolean>(true);
  const [loading, setLoading] = useState<boolean>(false);
  const [selectedProject, setSelectedProject] = useState<any>(null);

  const getConversationIdForContact = async (payload: { contactId: string, projectId: string }): Promise<string> => {
    try {
      const response = await fetch(`/api/messages/conversations/project`, {
        method: "POST", 
        body: JSON.stringify(payload)
      });

      if (!response.ok) {
        throw new Error(`Failed to fetch conversation: ${response.status}`);
      }

      const data = await response.json();

      return data.data.conversationId;
    } catch (error) {
      console.error('Error getting conversation ID:', error);
      throw error;
    }
  };

  useEffect(() => {
    const initConversation = async () => {
      const targetContactId = preselectedContactId ;
      let targetConversationId = propConversationId || "";

      if (!targetConversationId && targetContactId) {
        if (projectId) {
          let res = await getConversationIdForContact({
            projectId: projectId,
            contactId: targetContactId,
          });

          if(res) {
            targetConversationId = res;
          }

        }
      }

      if (targetConversationId && targetConversationId !== conversationId) {
        setSelectedContactId(targetContactId || undefined);
        setConversationId(targetConversationId);

        if (targetContactId) {
          await fetchContactDetails(targetContactId);
        }

        setShouldSelectFirst(false);
        setShowSidebar(false);

        if (projectId) setSelectedProjectId(projectId);
        if (projectStatus) setSelectedProjectStatus(projectStatus);
      }
    };

    initConversation();
  }, [preselectedContactId, contactId, projectId, projectStatus, propConversationId]);


  const fetchContactDetails = async (id: string) => {
    try {
      setLoading(true);
      const response = await fetch(`/api/messages/conversations/${id}`);
      if (response.ok) {
        const data = await response.json();
        if (data.data && data.data.contact) {
          setSelectedContact(data.data.contact);
        }
      }
    } catch (error) {
      console.error("Error fetching contact details:", error);
    } finally {
      setLoading(false);
    }
  };

  const handleSelectConversation = (
    conversationId: string,
    contact: User,
    project: any
  ) => {
    setSelectedContactId(contact.id);
    setConversationId(conversationId);

    if (project) {
      setSelectedProject(project);
      setSelectedProjectId(project.id);
      setSelectedProjectStatus(project.status);
    } else {
      setSelectedProjectId(projectId || "");
      setSelectedProjectStatus(projectStatus || "");
    }

    setSelectedContact(contact);
    setShouldSelectFirst(false);
    setShowSidebar(false);
  };

  const handleFirstConversationAvailable = (
    contactId: string,
    contact: User
  ) => {
    if (shouldSelectFirst && !conversationId) {
      setSelectedContactId(contactId);
      setConversationId(contactId);
      setSelectedContact(contact);
      setShouldSelectFirst(false);
      setShowSidebar(false);
    }
  };

  const handleBackClick = () => {
    setShowSidebar(true);
  };

  if (loading) {
    return <div className="chat-frame-wrap">Loading...</div>;
  }

  // Determine which ID to use for rendering MessageThread
  const activeConversationId = conversationId || selectedContactId;

  return (
    <div className="chat-frame-wrap" id="frame">
      <ConversationListNew
        onSelectConversation={handleSelectConversation}
        selectedContactId={selectedContactId}
        convId={conversationId}
        onFirstConversationAvailable={handleFirstConversationAvailable}
        shouldSelectFirst={shouldSelectFirst}
        showSidebar={showSidebar}
        selectedContact={selectedContact}
      />

      {activeConversationId ? (
        <MessageThread
          conversationId={activeConversationId}
          contact={selectedContact}
          projectId={selectedProjectId}
          projectStatus={selectedProjectStatus}
          handleBackClick={handleBackClick}
          showSidebar={showSidebar}
          selectedProject={selectedProject}
        />
      ) : (
        <div className="content">
          <div className="d-flex align-items-center justify-content-center h-100 text-muted">
            <div className="text-center">
              <img
                src="/assets/image/message-chat.svg"
                alt=""
                width="80"
                height="80"
                className="mb-3 opacity-50"
              />
              <h5>Select a conversation</h5>
              <p>Choose a conversation from the left to start messaging</p>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}

