# Repository Guidelines

## Project Structure & Module Organization
- Source: `src/` with App Router in `src/app/` (localized routes under `src/app/[locale]/`).
- UI: Reusable components in `src/components/`; hooks in `src/hooks/`; shared utils/config in `src/lib/`; types in `src/types/`.
- API: Route handlers in `src/app/api/*`.
- Assets: `public/` for static files.
- Data: Prisma schema and seeds in `prisma/`. Project scripts in `scripts/`.

## Build, Test, and Development Commands
```bash
npm run dev        # Start custom server (Next + WebSocket) on :3005
npm run dev:next   # Start Next dev server only (no custom WS)
npm run build      # Build Next.js for production
npm run start      # Start custom production server
npm run start:next # Start Next server only in production
npm run lint       # Run ESLint (Next + TS rules)

npm run db:generate # Generate Prisma client
npm run db:push     # Apply schema to DB
npm run db:seed     # Seed sample data
npm run db:studio   # Open Prisma Studio
```

## Coding Style & Naming Conventions
- Language: TypeScript; 2‑space indentation; prefer explicit types at module boundaries.
- React: Function components; components in `PascalCase` (e.g., `ProjectCard.tsx`).
- Hooks: `use*` prefix in `src/hooks/` (e.g., `useDebounce.ts`).
- Routes: Folder/segment names kebab- or lower-case under `src/app/`; colocate page/layout files per route.
- Linting: ESLint configured via `.eslintrc.json` (extends `next/core-web-vitals`, `react`, `@typescript-eslint`). Fix issues before PR.

## Testing Guidelines
- Framework: No test runner is configured yet. Validate changes with `npm run lint` and a production build (`npm run build`).
- If adding tests, prefer `*.spec.ts(x)` colocated near code or under `__tests__/`. Keep tests deterministic; mock network/DB.

## Commit & Pull Request Guidelines
- Commits: Use clear, imperative summaries (e.g., "Refactor ProjectsPage data fetching"). Group related changes; keep diffs focused.
- PRs: Include purpose, scope, and testing steps. Link related issues. Add screenshots/GIFs for UI changes. Call out DB schema changes (`prisma/schema.prisma`) and required env vars.

## Security & Configuration Tips
- Secrets: Never commit `.env*`; use `.env.local` for development. Required keys include DB (`DATABASE_URL`), NextAuth, and Stripe.
- Database: After schema edits, run `npm run db:generate` and `npm run db:push`, then reseed if needed.
- Realtime: WebSocket endpoint is `ws://localhost:3005/api/websocket` via `server.js`; be cautious when changing connection handling.
