
import { PrismaClient } from '@prisma/client';

const prisma = new PrismaClient();

async function main() {
    console.log('Checking for orphaned projects...');

    // Find projects where the categoryId does not exist in the Category table
    const orphanedProjects: any[] = await prisma.$queryRaw`
    SELECT id, title, categoryId
    FROM Project
    WHERE categoryId NOT IN (SELECT id FROM Category)
  `;

    console.log(`Found ${orphanedProjects.length} orphaned projects.`);

    if (orphanedProjects.length > 0) {
        const projectIds = orphanedProjects.map(p => p.id);
        const idsString = projectIds.map(id => `'${id}'`).join(',');

        console.log('Orphaned project IDs:', projectIds);

        // 1. Delete Messages related to these projects (directly or via Bids/Conversations)
        // Note: This is complex because messages can be linked to Bids or Conversations which are linked to Projects
        // We'll delete messages linked to the projects directly first
        console.log('Deleting related Messages...');
        await prisma.$executeRawUnsafe(`
      DELETE FROM Message 
      WHERE projectId IN (${idsString})
    `);

        // Also delete messages linked to Bids of these projects
        await prisma.$executeRawUnsafe(`
      DELETE FROM Message 
      WHERE bidId IN (SELECT id FROM Bid WHERE projectId IN (${idsString}))
    `);

        // Also delete messages linked to Conversations of these projects
        await prisma.$executeRawUnsafe(`
      DELETE FROM Message 
      WHERE conversationId IN (SELECT id FROM Conversation WHERE projectId IN (${idsString}))
    `);

        // 2. Delete Reviews
        console.log('Deleting related Reviews...');
        await prisma.$executeRawUnsafe(`
      DELETE FROM Review 
      WHERE projectId IN (${idsString})
    `);

        // 3. Delete Bids
        console.log('Deleting related Bids...');
        await prisma.$executeRawUnsafe(`
      DELETE FROM Bid 
      WHERE projectId IN (${idsString})
    `);

        // 4. Delete Conversations
        console.log('Deleting related Conversations...');
        await prisma.$executeRawUnsafe(`
      DELETE FROM Conversation 
      WHERE projectId IN (${idsString})
    `);

        // 5. Finally delete the Projects
        console.log('Deleting orphaned projects...');
        const deleted = await prisma.$executeRawUnsafe(`
      DELETE FROM Project
      WHERE id IN (${idsString})
    `);

        console.log(`Successfully deleted ${deleted} orphaned projects and their dependencies.`);
    } else {
        console.log('No orphaned projects found.');
    }
}

main()
    .catch((e) => {
        console.error(e);
        process.exit(1);
    })
    .finally(async () => {
        await prisma.$disconnect();
    });
