import { NextRequest, NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { generateJwtToken } from '@/lib/api/jwt';
import { prisma } from '@/lib/prisma';
import { sendMail } from '@/lib/send-mail';


export async function POST(request: NextRequest) {
    try {
        // const session = await getServerSession(authOptions);

        // if (!session?.user) {
        //     return NextResponse.json(
        //         { message: 'Unauthorized - Please login' },
        //         { status: 401 }
        //     );
        // }

        const body = await request.json();
        const { email, locale } = body;

        if (!email) {
            return NextResponse.json(
                { message: 'Email is required' },
                { status: 400 }
            );
        }

        // check if user with this email exists
        const existingUser = await prisma.user.findUnique({
            where: { email: email }
        });
        if (!existingUser) {
            return NextResponse.json(
                { message: 'If that email address is in our database, we will send you an email to reset your password' },
                { status: 200 }
            );
        }

        // generate a password reset token and it's expiration 
        const resetPaswordToken = await generateJwtToken({
            payload: { id: existingUser.id, email: existingUser.email },
            secret: process.env.JWT_SECRET_KEY as string,
            expiresIn: '30m'
        });

        const expirationTime = new Date(Date.now() + 30 * 60 * 1000);
        // save this token to DB 
        const user = await prisma.user.update({
            where: { id: existingUser.id },
            data: {
                passwordResetToken: resetPaswordToken,
                passwordResetAt: expirationTime
            }
        });

        const link = `${process.env.NEXTAUTH_URL}/${locale}/reset-password?token=${resetPaswordToken}`;


        const html = `<p>You requested a password reset. Click the link below to reset your password:</p>
        <p><a href="${link}">Reset Password</a></p>
        <p>This link will expire in 30 minutes.</p>
        <p>Copy this Link if that Link is not working: ${link}</p>
        <p>If you did not request this, please ignore this email.</p>
        `;

        const mail = await sendMail({
            sendTo: existingUser.email,
            subject: 'Password Reset Instructions',
            text: `You requested a password reset. Use the following link to reset your password: ${link}. This link will expire in 30 minutes. If you did not request this, please ignore this email.`,
            html: html
        });

        if (!mail) {
            return NextResponse.json(
                { message: 'Failed to send password reset email' },
                { status: 500 }
            );
        }


        // send a email with password reset link
        return NextResponse.json(
            { message: 'Password reset link has been sent to your email', link },
            { status: 200 }
        );

    } catch (error) {
        console.error('Change password error:', error);
        return NextResponse.json(
            { message: 'Internal server error' },
            { status: 500 }
        );
    }
}