import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/prisma';
import {
  getCategoryName,
  getCategoryDescription,
  parseLocale
} from '@/lib/category-utils';

export async function GET(req: NextRequest) {
  try {
    if (!prisma) {
      return NextResponse.json(
        { error: 'Database connection not available' },
        { status: 503 }
      );
    }

    const { searchParams } = new URL(req.url);
    const countryId = searchParams.get('countryId') || '';
    const localeParam = searchParams.get('locale');
    const locale = parseLocale(localeParam);

    // Get top 8 categories with project counts
    const categories = await prisma.category.findMany({
      where: {
        isActive: true
      },
      select: {
        id: true,
        name_en: true,
        name_fr: true,
        name_de: true,
        name_it: true,
        name_pt: true,
        name_es: true,
        description_en: true,
        description_fr: true,
        description_de: true,
        description_it: true,
        description_pt: true,
        description_es: true,
        slug: true,
        icon: true,
        _count: {
          select: {
            projects: {
              where: {
                countryId: countryId,
                status: 'OPEN'
              }
            }
          }
        }
      },
      orderBy: [
        {
          projects: {
            _count: 'desc'
          }
        },
        {
          name_en: 'asc'
        }
      ],
      take: 8
    });

    // Format the response with localized fields
    const formattedCategories = categories.map(category => ({
      id: category.id,
      name: getCategoryName(category, locale),
      slug: category.slug,
      description: getCategoryDescription(category, locale),
      icon: category.icon,
      image: `/assets/image/category/${category.icon}`,
      projectCount: category._count.projects
    }));

    return NextResponse.json({
      success: true,
      data: formattedCategories
    });

  } catch (error) {
    console.error('Error fetching featured categories:', error);

    return NextResponse.json(
      {
        error: error instanceof Error ? error.message : 'Failed to fetch featured categories',
        type: 'server_error'
      },
      { status: 500 }
    );
  }
}