import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/prisma';
import { getCategoryName, getCategoryDescription, parseLocale } from '@/lib/category-utils';

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const parentId = searchParams.get('parentId');
    const active = searchParams.get('active');
    const countryId = searchParams.get('countryId') || '';
    const FreelancerId = searchParams.get('fr') || '';

    if (FreelancerId) {
      const user = await prisma.user.findFirst({
        where: {
          id: FreelancerId,
          profile: {
            countryId: countryId,
          },
        },
        include: {
          profile: true
        }
      });
      if (!user) {
        return NextResponse.json(
          { message: 'Freelancer not found' },
          { status: 404 }
        );
      }
    }

    const whereClause: any = {};

    if (parentId) {
      whereClause.parentId = parentId;
    } else if (parentId === 'null') {
      whereClause.parentId = null;
    }

    if (active !== null) {
      whereClause.isActive = active === 'true';
    }

    // ✅ Freelancer subscription filter
    if (FreelancerId) {
      const existingSubscriptions = await prisma.categorySubscription.findMany({
        where: {
          userId: FreelancerId,
          status: 'ACTIVE',
          endDate: { gte: new Date() },
        },
        select: { categoryId: true },
      });

      const subscribedCategoryIds = existingSubscriptions.map((s) => s.categoryId);

      // Only filter if there are any active subscriptions
      if (subscribedCategoryIds.length > 0) {
        whereClause.id = { in: subscribedCategoryIds };
      } else {
        // No active subscriptions -> no categories will match
        whereClause.id = { in: [] };
      }
    }

    const categories = await prisma.category.findMany({
      where: whereClause,
      include: {
        children: {
          where: {
            isActive: true
          }
        },
        _count: {
          select: {
            projects: {
              where: {
                countryId: countryId,
                status: 'OPEN'
              }
            },
            profiles: true
          }
        }
      },
      orderBy: {
        name_en: 'asc'
      }
    });

    // Map to include name and description for frontend compatibility
    const localeParam = searchParams.get('locale');
    const locale = parseLocale(localeParam);

    const formattedCategories = categories.map(cat => ({
      ...cat,
      name: getCategoryName(cat, locale),
      description: getCategoryDescription(cat, locale),
      children: cat.children?.map(child => ({
        ...child,
        name: getCategoryName(child, locale),
        description: getCategoryDescription(child, locale)
      }))
    }));

    return NextResponse.json(formattedCategories);

  } catch (error) {
    console.error('Categories fetch error:', error);
    return NextResponse.json(
      { message: 'Internal server error' },
      { status: 500 }
    );
  }
} 