import { NextRequest, NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';
import { getCategoryName, parseLocale } from '@/lib/category-utils';

export async function GET(req: NextRequest) {
  try {
    const session = await getServerSession(authOptions);

    if (!session?.user?.id) {
      return NextResponse.json({ success: false, error: 'Unauthorized' }, { status: 401 });
    }

    if (!prisma) {
      return NextResponse.json({ success: true, data: [] });
    }

    const subs = await prisma.categorySubscription.findMany({
      where: {
        userId: session.user.id,
        status: 'ACTIVE',
        endDate: { gte: new Date() },
      },
      select: {
        category: {
          select: {
            id: true,
            name_en: true,
            name_fr: true,
            name_de: true,
            name_it: true,
            name_pt: true,
            name_es: true,
            slug: true,
            icon: true
          },
        },
      },
      orderBy: { createdAt: 'desc' },
    });

    const localeParam = req.nextUrl.searchParams.get('locale');
    const locale = parseLocale(localeParam);

    const categories = subs
      .map((s: any) => ({
        ...s.category,
        name: getCategoryName(s.category, locale)
      }))
      .filter(Boolean)
      .sort((a: any, b: any) => a.name.localeCompare(b.name));

    return NextResponse.json({ success: true, data: categories });
  } catch (error) {
    console.error('Error fetching subscribed categories:', error);
    return NextResponse.json(
      { success: false, error: 'Failed to fetch subscribed categories' },
      { status: 500 }
    );
  }
}

