import { NextRequest, NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';
import { writeFile, mkdir } from 'fs/promises';
import { join } from 'path';
import { existsSync } from 'fs';

export async function POST(req: NextRequest) {
  try {
    if (!prisma) {
      return NextResponse.json(
        { error: 'Database connection not available' },
        { status: 503 }
      );
    }

    const session = await getServerSession(authOptions);

    if (!session?.user?.email) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }

    const formData = await req.formData();
    const file = formData.get('profileImage') as File;

    if (!file) {
      return NextResponse.json({ error: 'No file provided' }, { status: 400 });
    }

    const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png'];
    if (!allowedTypes.includes(file.type)) {
      return NextResponse.json(
        { error: 'Invalid file type. Only JPG and PNG files are allowed.' },
        { status: 400 }
      );
    }

    const maxSize = 5 * 1024 * 1024; // 5MB
    if (file.size > maxSize) {
      return NextResponse.json(
        { error: 'File size exceeds 5MB limit' },
        { status: 400 }
      );
    }

    const uploadsDir = join(process.cwd(), 'public', 'uploads', 'profiles');
    if (!existsSync(uploadsDir)) {
      await mkdir(uploadsDir, { recursive: true });
    }

    const timestamp = Date.now();
    const randomString = Math.random().toString(36).substring(2, 15);
    const fileExtension = file.name.split('.').pop() || 'jpg';
    const fileName = `profile_${session.user.email}_${timestamp}_${randomString}.${fileExtension}`;
    const filePath = join(uploadsDir, fileName);

    const bytes = await file.arrayBuffer();
    const buffer = Buffer.from(bytes);
    await writeFile(filePath, buffer);

    const imageUrl = `/uploads/profiles/${fileName}`;

    // Find user by email
    const userByEmail = await prisma.user.findUnique({
      where: { email: session.user.email },
      select: { id: true },
    });

    if (!userByEmail) {
      return NextResponse.json(
        { error: 'User not found in database' },
        { status: 404 }
      );
    }

    let updatedUser;

    try {
      // Try updating user table directly
      updatedUser = await prisma.user.update({
        where: { id: userByEmail.id },
        data: {
          image: imageUrl,
          updatedAt: new Date(),
        },
        select: {
          id: true,
          image: true,
        },
      });
    } catch (dbError: any) {
      console.warn('User update failed, trying profile update...', dbError?.code);

      // If the user table update fails, upsert the profile record
      updatedUser = await prisma.profile.upsert({
        where: { userId: userByEmail.id },
        update: { profileImage: imageUrl },
        create: {
          userId: userByEmail.id,
          profileImage: imageUrl,
          firstName: '',
          lastName: '',
          phone: '',
          country: '',
          city: '',
          zipCode: '',
          user: {
            connect: { id: userByEmail.id }, // ✅ FIX: required relation
          },
        },
      });
    }

    // Optional: log upload activity
    try {
      console.log(`Profile image updated for ${session.user.email}: ${fileName}`);
      // Uncomment when userActivity model is ready
      // await prisma.userActivity.create({
      //   data: {
      //     userId: userByEmail.id,
      //     action: 'PROFILE_IMAGE_UPDATED',
      //     details: `Image: ${fileName}`,
      //     ipAddress: req.headers.get('x-forwarded-for') || req.headers.get('x-real-ip') || 'unknown',
      //     userAgent: req.headers.get('user-agent') || 'unknown',
      //   },
      // });
    } catch (logErr) {
      console.error('Failed to log upload activity:', logErr);
    }

    return NextResponse.json({
      success: true,
      data: {
        imageUrl,
        fileName,
      },
      message: 'Profile image uploaded successfully',
    });
  } catch (error: any) {
    console.error('Error uploading profile image:', error);

    if (
      error.message?.includes('connect') ||
      error.message?.includes('database') ||
      error.message?.includes('relation') ||
      error.message?.includes('table')
    ) {
      const mockImageUrl = `/assets/image/profile_${Date.now()}.jpg`;
      return NextResponse.json({
        success: true,
        data: {
          imageUrl: mockImageUrl,
          fileName: `mock_profile_${Date.now()}.jpg`,
        },
        message: 'Profile image upload simulated (database not available)',
      });
    }

    return NextResponse.json(
      {
        error: error.message || 'Failed to upload profile image',
        type: 'server_error',
      },
      { status: 500 }
    );
  }
}

export const config = {
  api: {
    bodyParser: {
      sizeLimit: '5mb',
    },
  },
};
