import { NextRequest, NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';

export async function POST(
  req: NextRequest,
  
) {
  const { projectId, contactId } = await req.json();

  try {
    const session = await getServerSession(authOptions);

    if (!session?.user?.email) {
      return NextResponse.json(
        { error: 'Unauthorized' },
        { status: 401 }
      );
    }

    const user = await prisma.user.findUnique({
      where: { email: session.user.email }
    });

    if (!user) {
      return NextResponse.json(
        { error: 'User not found' },
        { status: 404 }
      );
    }

    // Find existing conversation between these users for this project
    const conversation = await prisma.conversation.findFirst({
      where: {
        OR: [
          {
            buyerId: user.id,
            sellerId: contactId,
            projectId: projectId
          },
          {
            buyerId: contactId,
            sellerId: user.id,
            projectId: projectId
          }
        ]
      },
      select: {
        id: true // Only return the conversationId
      }
    });


    return NextResponse.json({
      success: true,
      data: {
        conversationId: conversation.id
      }
    });

  } catch (error) {
    console.error('Error finding/conversation:', error);
    return NextResponse.json(
      {
        error: error instanceof Error ? error.message : 'Failed to process conversation',
        type: 'server_error'
      },
      { status: 500 }
    );
  }
}