import { NextRequest, NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';
import { getCategoryName, parseLocale, CATEGORY_MULTILINGUAL_FIELDS } from '@/lib/category-utils';

export async function GET(req: NextRequest) {
  try {
    const session = await getServerSession(authOptions);

    if (!session?.user?.email) {
      return NextResponse.json(
        { error: 'Unauthorized' },
        { status: 401 }
      );
    }

    const { searchParams } = new URL(req.url);
    const category = searchParams.get('category');
    const search = searchParams.get('search');
    const page = parseInt(searchParams.get('page') || '1');
    const limit = parseInt(searchParams.get('limit') || '10');
    const offset = (page - 1) * limit;

    if (!prisma) {
      return NextResponse.json(
        { error: 'Database connection not available' },
        { status: 503 }
      );
    }

    try {
      // Build where clause
      const whereClause: any = {
        status: 'OPEN', // Only show open projects (not awarded, cancelled, or completed)
        publishedAt: {
          not: null
        }
      };

      // Exclude projects where user has already bid
      whereClause.bids = {
        none: {
          freelancerId: session.user.email
        }
      };

      if (category) {
        whereClause.categoryId = category;
      }

      if (search) {
        whereClause.OR = [
          { title: { contains: search } },
          { description: { contains: search } }
        ];
      }

      // Get projects
      const projects = await prisma.project.findMany({
        where: whereClause,
        include: {
          category: {
            select: {
              ...CATEGORY_MULTILINGUAL_FIELDS,
              slug: true
            }
          },
          client: {
            select: {
              profile: {
                select: {
                  firstName: true,
                  lastName: true
                }
              }
            }
          },
          bids: {
            select: {
              id: true
            }
          }
        },
        orderBy: {
          createdAt: 'desc'
        },
        skip: offset,
        take: limit
      });

      // Get total count
      const totalCount = await prisma.project.count({
        where: whereClause
      });

      const localeParam = searchParams.get('locale');
      const locale = parseLocale(localeParam);

      const formattedProjects = projects.map(project => ({
        id: project.id,
        title: project.title,
        description: project.description,
        budget: project.budget,
        duration: project.duration,
        skills: project.skills ? JSON.parse(project.skills) : [],
        categoryName: getCategoryName(project.category, locale),
        categorySlug: project.category.slug,
        clientName: `${project.client.profile?.firstName || ''} ${project.client.profile?.lastName || ''}`.trim() || 'Anonymous',
        bidCount: project.bids.length,
        publishedAt: project.publishedAt?.toISOString(),
        timeAgo: getTimeAgo(project.publishedAt || project.createdAt)
      }));

      return NextResponse.json({
        success: true,
        data: {
          projects: formattedProjects,
          pagination: {
            page,
            limit,
            totalCount,
            totalPages: Math.ceil(totalCount / limit),
            hasNext: offset + limit < totalCount,
            hasPrev: page > 1
          }
        }
      });

    } catch (dbError: any) {
      console.error('Database error:', dbError);

      return NextResponse.json(
        {
          error: 'Database operation failed',
          message: 'Unable to fetch projects due to a database error. Please try again later.',
          details: process.env.NODE_ENV === 'development' ? dbError.message : undefined
        },
        { status: 500 }
      );
    }

  } catch (error) {
    console.error('Error fetching projects:', error);
    return NextResponse.json(
      {
        error: error instanceof Error ? error.message : 'Failed to fetch projects',
        type: 'server_error'
      },
      { status: 500 }
    );
  }
}

function getTimeAgo(date: Date): string {
  const now = new Date();
  const diffInMs = now.getTime() - date.getTime();
  const diffInMinutes = Math.floor(diffInMs / (1000 * 60));
  const diffInHours = Math.floor(diffInMinutes / 60);
  const diffInDays = Math.floor(diffInHours / 24);

  if (diffInMinutes < 1) return 'Just now';
  if (diffInMinutes < 60) return `${diffInMinutes} minute${diffInMinutes > 1 ? 's' : ''} ago`;
  if (diffInHours < 24) return `${diffInHours} hour${diffInHours > 1 ? 's' : ''} ago`;
  if (diffInDays < 7) return `${diffInDays} day${diffInDays > 1 ? 's' : ''} ago`;
  return date.toLocaleDateString();
}
