"use client";

import Link from "next/link";
import Image from "next/image";
import { usePathname, useRouter } from "next/navigation";
import { useSession, signOut } from "next-auth/react";
import { useState, useEffect } from "react";
import NotificationPanel from "./notifications/NotificationPanel";
import MessagesDropdown from "./messages/MessagesDropdown";
import { useRememberMe } from "@/hooks/useRememberMe";
import { useTranslations } from "next-intl";
import { useCountry } from "@/context/domainContext";

interface HeaderProps {
  locale: string;
}

export default function Header({ locale }: HeaderProps) {
  const pathname = usePathname();
  const router = useRouter();
  const { data: session, status } = useSession();
  const { clearEmail } = useRememberMe();

  const handleSignOut = async () => {
    clearEmail();
    await signOut({ callbackUrl: `/${locale}` });
  };

  const userRole = (session?.user as any)?.role;
  const isClient = userRole === "CLIENT";

  const t = useTranslations();
  const baseUrl = `${process.env.NEXTAUTH_URL || "http://localhost:3000"}`;

  const defaultImage = "/assets/image/loginuser.png";

  const imageSrc = session?.user?.image
    ? `${baseUrl}${session?.user?.image}`
    : defaultImage;

  const [currentPath, setCurrentPath] = useState(pathname);

  useEffect(() => {
    setCurrentPath(pathname);
  }, [pathname]);
  const { country } = useCountry();
  console.log("country22222",country)
  return (
    <>
      <header>
        <nav className="navbar navbar-expand-lg">
          <div className="container-fluid">
            <Link 
              href={`/${locale}`} 
              className="navbar-brand p-0"
            >
              <Image
                src={country?.logoHeader || "/assets/image/logo.png"}
                className="img-fluid"
                alt="logo"
                width={128}
                height={59}
                priority
                onError={(e) => {
                  e.currentTarget.onerror = null; 
                  e.currentTarget.src = "/assets/image/logo.png";
                }}
              />
            </Link>
            {currentPath && (
              <div
                className="collapse navbar-collapse"
                id="navbarSupportedContent"
                key={pathname}
              >
                <ul className="navbar-nav me-auto mb-lg-0">
                  <li className="nav-item">
                    <Link
                      className={`nav-link ${
                        pathname === `/${locale}` || pathname === `/${locale}/`
                          ? "active"
                          : ""
                      }`}
                      href={`/${locale}`}
                    >
                      {t("header.home")}
                    </Link>
                  </li>
                  <li className="nav-item">
                    <Link
                      className={`nav-link ${
                        pathname.includes("/how-it-works") ? "active" : ""
                      }`}
                      href={`/${locale}/how-it-works`}
                    >
                      {t("header.how_it_works")}
                    </Link>
                  </li>
                  <li className="nav-item">
                    <Link
                      className={`nav-link ${
                        pathname.includes("/projects") &&
                        !pathname.includes("/my-projects") &&
                        !pathname.includes("/post-project")
                          ? "active"
                          : ""
                      }`}
                      href={`/${locale}/projects`}
                    >
                      {t("header.find_projects")}
                    </Link>
                  </li>
                  <li className="nav-item">
                    <Link
                      className={`nav-link ${
                        pathname.includes("/categories") ? "active" : ""
                      }`}
                      href={`/${locale}/categories`}
                    >
                      {t("header.service_category")}
                    </Link>
                  </li>
                  <li className="nav-item">
                    <Link
                      href={session?.user ? "/freelancers" : "/login"}
                      className={`nav-link ${
                        pathname.includes("/freelancers") &&
                        !pathname.includes("/freelancer/")
                          ? "active"
                          : ""
                      }`}
                    >
                      {t("header.freelancers")}
                    </Link>
                  </li>
                </ul>
                <div className="d-flex button-gropup align-items-center login-signup-nav">
                  {status === "authenticated" && session?.user ? (
                    <>
                      {isClient && (
                        <div className="post-project-link">
                          <Link
                            href={`/${locale}/client/post-project`}
                            className="custom-btn primary-outline text-uppercase"
                          >
                            {t("header.post_project")}
                          </Link>
                        </div>
                      )}
                    </>
                  ) : (
                    <>
                      <div className="post-project-link">
                        <Link
                          href={`/${locale}/client/post-project`}
                          className="custom-btn primary-outline text-uppercase"
                        >
                          {t("header.post_project")}
                        </Link>
                      </div>
                      <Link
                        href={`/${locale}/login`}
                        className={`nav-btn ${
                          pathname.includes("/login") ? "active" : ""
                        }`}
                      >
                        {t("header.login")}
                      </Link>
                      <Link
                        href={`/${locale}/signup`}
                        className={`nav-btn ${
                          pathname.includes("/signup") ? "active" : ""
                        }`}
                      >
                        {t("header.signup")}
                      </Link>
                    </>
                  )}
                </div>
              </div>
            )}
            <div className="d-flex align-items-center gap-2">
              {status === "authenticated" && session?.user && (
                <div className="d-flex button-gropup loginuser-nav ps-4">
                  <NotificationPanel />
                  <MessagesDropdown />
                  <div className="language-selector userlogin-dropdown">
                    <div className="nav-item dropdown language-dropdown">
                      <button type="button" className="nav-link select-lang">
                        <Image
                          src={imageSrc}
                          alt="user"
                          width={0}
                          height={0}
                          sizes="100vw"
                          style={{ width: "40px", height: "40px" }}
                          className="me-2"
                          priority
                          onError={(e) => {
                            e.currentTarget.onerror = null; // Prevent infinite loop
                            e.currentTarget.src = "/assets/image/profile03.png";
                          }}
                        />
                        <div>{session.user?.name || "User"}</div>
                      </button>
                      <ul>
                        <li>
                          <Link
                            className={
                              pathname.includes("/profile") ? "active" : ""
                            }
                            href={`/${locale}/${userRole.toLowerCase()}/profile`}
                          >
                            <Image
                              src="/assets/image/profile-circle.svg"
                              alt="My Profile"
                              width={0}
                              height={0}
                              sizes="100vw"
                              style={{ width: "16px", height: "16px" }}
                              className="me-2"
                              priority
                            />
                            {t("header.profile")}
                          </Link>
                        </li>
                        <li>
                          <Link
                            className={
                              pathname.includes("/message") ? "active" : ""
                            }
                            href={`/${locale}/${userRole.toLowerCase()}/messages`}
                          >
                            <Image
                              src="/assets/image/user-edit.svg"
                              alt="Messages"
                              width={0}
                              height={0}
                              sizes="100vw"
                              style={{ width: "16px", height: "16px" }}
                              className="me-2"
                              priority
                            />
                            {t("header.message")}
                          </Link>
                        </li>
                        <li>
                          <Link
                            className={
                              pathname.includes(
                                isClient ? "/my-projects" : "/my-projects"
                              )
                                ? "active"
                                : ""
                            }
                            href={
                              isClient
                                ? `/${locale}/client/my-projects`
                                : `/${locale}/freelancer/my-projects`
                            }
                          >
                            <Image
                              src="/assets/image/my-project.svg"
                              alt="My Projects"
                              width={0}
                              height={0}
                              sizes="100vw"
                              style={{ width: "16px", height: "16px" }}
                              className="me-2"
                              priority
                            />
                            {isClient ? "My Projects" : "My Proposals"}
                          </Link>
                        </li>
                        <li>
                          <button
                            type="button"
                            onClick={handleSignOut}
                            className="dropdown-item"
                            style={{ paddingLeft: "15px" }}
                          >
                            <Image
                              src="/assets/image/logout.svg"
                              alt="Logout"
                              width={0}
                              height={0}
                              sizes="100vw"
                              style={{ width: "16px", height: "16px" }}
                              className="me-2"
                            />
                            {t("header.logout")}
                          </button>
                        </li>
                      </ul>
                    </div>
                  </div>
                </div>
              )}
              {currentPath && (
                <button
                  className="navbar-toggler"
                  type="button"
                  data-bs-toggle="collapse"
                  data-bs-target="#navbarSupportedContent"
                  aria-controls="navbarSupportedContent"
                  aria-expanded="false"
                  aria-label="Toggle navigation"
                  key={pathname}
                >
                  <span className="navbar-toggler-icon"></span>
                </button>
              )}
            </div>
          </div>
        </nav>
      </header>

      <style jsx>{`
        .post-project-link a {
          font-weight: 500;
        }

        .primary-outline,
        .primary-fill:hover {
          background-color: var(--color-white);
          color: var(--color-primary);
          border-color: var(--color-primary);
        }
        .custom-btn {
          display: flex;
          align-items: center;
          justify-content: center;
          gap: 8px;
          border: 1px solid;
          padding: 7px 16px;
          transition: all 0.4s ease-in-out;
          cursor: pointer;
          font-size: var(--font-regular);
          line-height: 24px;
          border-radius: 4px;
          text-transform: capitalize;
          width: auto !important;
        }
        a {
          text-decoration: none !important;
          transition: all 0.4s ease-in-out;
          color: var(--color-primary);
        }
        header {
          width: 100%;
          position: fixed;
          top: 0;
          left: 0;
          z-index: 91;
          background-color: #ffffff;
        }
      `}</style>
    </>
  );
}
