'use client';

import Link from 'next/link';
import { useState } from 'react';
import { Card, Row, Col, Badge, Button, Table } from 'react-bootstrap';

interface FreelancerDashboardProps {
  user: any;
  stats: {
    totalBids: number;
    acceptedBids: number;
    pendingBids: number;
    activeProjects: number;
    activeSubscription: number;
  };
  recentProjects: any[];
  locale: string;
}

export default function FreelancerDashboard({ user, stats, recentProjects, locale }: FreelancerDashboardProps) {
  const [activeTab, setActiveTab] = useState('overview');

  const getStatusBadge = (status: string) => {
    const statusConfig = {
      PENDING: { variant: 'warning', text: 'Pending' },
      ACCEPTED: { variant: 'success', text: 'Accepted' },
      REJECTED: { variant: 'danger', text: 'Rejected' },
      WITHDRAWN: { variant: 'secondary', text: 'Withdrawn' }
    };
    
    const config = statusConfig[status as keyof typeof statusConfig] || { variant: 'secondary', text: status };
    return <Badge bg={config.variant}>{config.text}</Badge>;
  };

  const formatDate = (date: string) => {
    return new Date(date).toLocaleDateString();
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'EUR'
    }).format(amount);
  };

  return (
    <div className="dashboard-page">
      <div className="container-fluid">
        <div className="row">
          <div className="col-12">
            <div className="dashboard-header">
              <h1>Freelancer Dashboard</h1>
              <p>Welcome back, {user.profile?.firstName || user.email}!</p>
            </div>
          </div>
        </div>

        {/* Stats Cards */}
        <Row className="mb-4">
          <Col md={3}>
            <Card className="stat-card">
              <Card.Body>
                <div className="stat-content">
                  <div className="stat-number">{stats.totalBids}</div>
                  <div className="stat-label">Total Bids</div>
                </div>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="stat-card">
              <Card.Body>
                <div className="stat-content">
                  <div className="stat-number">{stats.acceptedBids}</div>
                  <div className="stat-label">Accepted Bids</div>
                </div>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="stat-card">
              <Card.Body>
                <div className="stat-content">
                  <div className="stat-number">{stats.activeProjects}</div>
                  <div className="stat-label">Active Projects</div>
                </div>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="stat-card">
              <Card.Body>
                <div className="stat-content">
                  <div className="stat-number">{stats.activeSubscription ? 'Active' : 'Inactive'}</div>
                  <div className="stat-label">Subscription</div>
                </div>
              </Card.Body>
            </Card>
          </Col>
        </Row>

        {/* Quick Actions */}
        <Row className="mb-4">
          <Col>
            <Card>
              <Card.Body>
                <h5>Quick Actions</h5>
                <div className="d-flex gap-2 flex-wrap">
                  <Link href={`/${locale}/freelancer/projects`}>
                    <Button variant="primary">Find Projects</Button>
                  </Link>
                  <Link href={`/${locale}/freelancer/profile`}>
                    <Button variant="outline-primary">Edit Profile</Button>
                  </Link>
                  {!stats.activeSubscription && (
                    <Link href={`/${locale}/freelancer/subscription`}>
                      <Button variant="outline-warning">Get Subscription</Button>
                    </Link>
                  )}
                </div>
              </Card.Body>
            </Card>
          </Col>
        </Row>

        {/* Recent Bids */}
        <Row className="mb-4">
          <Col>
            <Card>
              <Card.Header>
                <h5 className="mb-0">Recent Bids</h5>
              </Card.Header>
              <Card.Body>
                {user.freelancerBids.length > 0 ? (
                  <Table responsive>
                    <thead>
                      <tr>
                        <th>Project</th>
                        <th>Client</th>
                        <th>Your Bid</th>
                        <th>Status</th>
                        <th>Submitted</th>
                        <th>Actions</th>
                      </tr>
                    </thead>
                    <tbody>
                      {user.freelancerBids.slice(0, 5).map((bid: any) => (
                        <tr key={bid.id}>
                          <td>
                            <div>
                              <strong>{bid.project.title}</strong>
                              <br />
                              <small className="text-muted">
                                {bid.project.description.substring(0, 50)}...
                              </small>
                            </div>
                          </td>
                          <td>{bid.project.client.profile?.firstName || bid.project.client.email}</td>
                          <td>{formatCurrency(bid.amount)}</td>
                          <td>{getStatusBadge(bid.status)}</td>
                          <td>{formatDate(bid.createdAt)}</td>
                          <td>
                            <Link href={`/${locale}/freelancer/projects/${bid.project.id}`}>
                              <Button size="sm" variant="outline-primary">View</Button>
                            </Link>
                          </td>
                        </tr>
                      ))}
                    </tbody>
                  </Table>
                ) : (
                  <div className="text-center py-4">
                    <p className="text-muted">No bids yet</p>
                    <Link href={`/${locale}/freelancer/projects`}>
                      <Button variant="primary">Find Projects to Bid On</Button>
                    </Link>
                  </div>
                )}
              </Card.Body>
            </Card>
          </Col>
        </Row>

        {/* Recommended Projects */}
        <Row>
          <Col>
            <Card>
              <Card.Header>
                <h5 className="mb-0">Recommended Projects</h5>
              </Card.Header>
              <Card.Body>
                {recentProjects.length > 0 ? (
                  <Table responsive>
                    <thead>
                      <tr>
                        <th>Project</th>
                        <th>Client</th>
                        <th>Budget</th>
                        <th>Category</th>
                        <th>Posted</th>
                        <th>Actions</th>
                      </tr>
                    </thead>
                    <tbody>
                      {recentProjects.map((project: any) => (
                        <tr key={project.id}>
                          <td>
                            <div>
                              <strong>{project.title}</strong>
                              <br />
                              <small className="text-muted">
                                {project.description.substring(0, 50)}...
                              </small>
                            </div>
                          </td>
                          <td>{project.client.profile?.firstName || project.client.email}</td>
                          <td>{formatCurrency(project.budget)}</td>
                          <td>{project.category.name}</td>
                          <td>{formatDate(project.createdAt)}</td>
                          <td>
                            {project.bids.length > 0 ? (
                              <Badge bg="info">Already Bid</Badge>
                            ) : (
                              <Link href={`/${locale}/freelancer/projects/${project.id}`}>
                                <Button size="sm" variant="primary">Bid Now</Button>
                              </Link>
                            )}
                          </td>
                        </tr>
                      ))}
                    </tbody>
                  </Table>
                ) : (
                  <div className="text-center py-4">
                    <p className="text-muted">No recommended projects</p>
                    <Link href={`/${locale}/freelancer/projects`}>
                      <Button variant="primary">Browse All Projects</Button>
                    </Link>
                  </div>
                )}
              </Card.Body>
            </Card>
          </Col>
        </Row>
      </div>
    </div>
  );
} 