'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { maskEmail, maskMobileNumber, maskName } from '@/lib/utils/html-to-text';

interface RelatedFreelancersProps {
  countryId: string | null;
  currentFreelancerId: string;
  locale: string;
}

export default function RelatedFreelancers({ countryId, currentFreelancerId, locale }: RelatedFreelancersProps) {
  const [freelancers, setFreelancers] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const fetchRelatedFreelancers = async () => {
      try {
        setLoading(true);
        setError(null);

        // Call the freelancers API with the same parameters as the list page
        const apiUrl = new URL('/api/freelancers', window.location.origin);
        apiUrl.searchParams.set('sort', 'newest');
        apiUrl.searchParams.set('page', '1');

        if (countryId) {
          apiUrl.searchParams.set('location', countryId);
        }

        const response = await fetch(apiUrl.toString());

        if (!response.ok) {
          throw new Error(`API call failed: ${response.status}`);
        }

        const data = await response.json();

        // Filter out the current freelancer and take only 2
        const relatedFreelancers = data.freelancers
          .filter((freelancer: any) => freelancer.id !== currentFreelancerId)
          .slice(0, 2);

        setFreelancers(relatedFreelancers);
      } catch (err) {
        console.error('Error fetching related freelancers:', err);
        setError('Failed to load related freelancers');
      } finally {
        setLoading(false);
      }
    };

    fetchRelatedFreelancers();
  }, [countryId, currentFreelancerId]);
  const getAverageRating = (reviews: any[]) => {
    if (!reviews || reviews.length === 0) return 0;
    const total = reviews.reduce((sum, review) => sum + review.rating, 0);
    return (total / reviews.length);
  };

  const renderStars = (rating: number) => {
    const fullStars = Math.floor(rating);
    const hasHalfStar = rating % 1 >= 0.5;
    const stars = [];

    for (let i = 0; i < fullStars; i++) {
      stars.push(
        <svg key={i} width="13" height="12" viewBox="0 0 13 12" fill="none" xmlns="http://www.w3.org/2000/svg">
          <path d="M10.2062 12L6.5 9.96162L2.79241 12L3.50213 7.67959L0.5 4.62053L4.64478 3.99059L6.5 0L8.35522 3.99059L12.5 4.62053L9.49787 7.68107L10.2062 12Z" fill="#FB6107" />
        </svg>
      );
    }

    if (hasHalfStar) {
      stars.push(
        <svg key="half" width="13" height="12" viewBox="0 0 13 12" fill="none" xmlns="http://www.w3.org/2000/svg">
          <path d="M10.2062 12L6.5 9.96162L2.79241 12L3.50213 7.67959L0.5 4.62053L4.64478 3.99059L6.5 0L8.35522 3.99059L12.5 4.62053L9.49787 7.68107L10.2062 12Z" fill="#FB6107" opacity="0.5" />
        </svg>
      );
    }

    return stars;
  };

  if (loading) {
    return (
      <section className="related-freelancers py-60">
        <div className="container">
          <div className="row">
            <div className="col-12">
              <div className="section-title text-center mb-5">
                <h2>Related Freelancers</h2>
                <p>Loading related freelancers...</p>
              </div>
            </div>
          </div>
        </div>
      </section>
    );
  }

  if (error || !Array.isArray(freelancers) || freelancers.length === 0) {
    return null; // Don't show the section if there's an error or no freelancers
  }

  return (
    <section className="related-freelancers py-60">
      <div className="container">
        <div className="row">
          <div className="col-12">
            <div className="section-title text-center mb-5">
              <h2>Related Freelancers</h2>
              <p>Discover other talented freelancers you might be interested in</p>
            </div>
          </div>
        </div>
        <div className="row">
          <div className="col-12">
            <div className="freelancers-listing">
              {Array.isArray(freelancers) && freelancers.map((freelancer: any) => {
                const avgRating = getAverageRating(freelancer.receivedReviews);
                const reviewCount = freelancer.receivedReviews?.length || 0;

                return (
                  <div key={freelancer.id} className="freelancers-card">
                    <div className="fc-detail">
                      <div className="fc-profile d-flex align-items-center">
                        <div className="fc-img">
                          {freelancer.profileImage || freelancer.profile?.profileImage ? (
                            <img
                              src={freelancer.profileImage || freelancer.profile?.profileImage}
                              alt="Profile"
                              onError={(e) => {
                                e.currentTarget.onerror = null; // Prevent infinite loop
                                e.currentTarget.src = "/assets/image/profile03.png";
                              }}
                            />
                          ) : (
                            <img src="/assets/image/user.png" alt="Profile" />
                          )}
                        </div>
                        <div className="fc-content">
                          <h5>
                            {maskName(
                              freelancer.name ||
                              `${freelancer.profile?.firstName || ''} ${freelancer.profile?.lastName || ''}`.trim() ||
                              freelancer.email ||
                              'User'
                            )}
                          </h5>
                          <div className="d-flex align-items-center gap-1">
                            <span className="star-icon">
                              {renderStars(avgRating)}
                            </span>
                            {
                              (avgRating > 0 || reviewCount > 0) ? (
                                <>
                                  {avgRating.toFixed(1)} ({reviewCount}+ Reviews)
                                </>
                              ) : (
                                <>
                                <svg width="13" height="12" viewBox="0 0 13 12" fill="#FB6107" stroke="#FB6107" strokeWidth="1">
                                  <path d="M10.2062 12L6.5 9.96162L2.79241 12L3.50213 7.67959L0.5 4.62053L4.64478 3.99059L6.5 0L8.35522 3.99059L12.5 4.62053L9.49787 7.68107L10.2062 12Z"/>
                                </svg>
                                 {" "} <span>No review yet</span>
                                </>
                              )
                            }
                          </div>
                        </div>
                      </div>
                      <div className="fc-dis">
                        <p>
                          {freelancer.profile?.title || freelancer.bio?.substring(0, 60) || 'Creative Professional'}
                        </p>
                      </div>
                      <div className="divider"></div>
                      <ul className="fc-contact">
                        <li className="fc-text">
                          <div className="icon">
                            <img src="/assets/image/gmail.svg" alt="gmail" />
                          </div>
                          {freelancer.email ? maskEmail(freelancer.email) : ''}
                        </li>
                        {freelancer.profile?.phone && (
                          <li className="fc-text">
                            <div className="icon">
                              <img src="/assets/image/call.svg" alt="call" />
                            </div>
                            {freelancer.profile.phone ? maskMobileNumber(freelancer.profile.phone) : ''}
                          </li>
                        )}
                        <li className="fc-text">
                          <div className="icon">
                            <img src="/assets/image/location.svg" alt="location" />
                          </div>
                          {freelancer.profile?.country?.name || 'Location not specified'}
                        </li>
                      </ul>
                    </div>
                    <div className="freelancers-action button-group d-flex gap-2 flex-wrap">
                      <Link href={`/${locale}/freelancers/${freelancer.id}`} className="custom-btn primary-outline col text-nowrap">
                        View Profile
                      </Link>
                    </div>
                  </div>
                );
              })}
            </div>
          </div>
        </div>
      </div>
    </section>
  );
}
