// context/CountryContext.tsx
"use client";

import { createContext, useContext, ReactNode, useState, useEffect } from "react";

export interface Country {
  id: string;
  name: string;
  code: string;
  domain: string;
  languages: string[];
  paymentGateways: string[];
  // add other optional fields if needed
}

interface CountryContextType {
  country: Country | null;
  setCountry: (country: Country) => void;
  countries: Country[];
  loading: boolean;
  refreshCountries: () => Promise<void>;
}

const CountryContext = createContext<CountryContextType | undefined>(undefined);

interface CountryProviderProps {
  children: ReactNode;
  initialCountry?: Country | null; // optional initial country
}

export const CountryProvider = ({ children, initialCountry = null }: CountryProviderProps) => {
  const [country, setCountry] = useState<Country | null>(initialCountry);
  const [countries, setCountries] = useState<Country[]>([]);
  const [loading, setLoading] = useState(true);

  // Helper to safely parse strings to arrays
  const safeParseArray = (value: any): string[] => {
    if (!value) return [];
    if (Array.isArray(value)) return value;
    try {
      return JSON.parse(value);
    } catch {
      return [];
    }
  };

  // Fetch countries from API
  const fetchCountries = async () => {
    setLoading(true);
    try {
      const res = await fetch("/api/countries");
      if (!res.ok) throw new Error("Failed to fetch countries");
      const data = await res.json();

      // Normalize data
      const normalizedData: Country[] = data.map((d: any) => ({
        ...d,
        languages: safeParseArray(d.languages),
        paymentGateways: safeParseArray(d.paymentGateways),
      }));

      setCountries(normalizedData);

      // Set default country if not already set
      if (!country && normalizedData.length > 0) {
        setCountry(normalizedData[0]);
      }
    } catch (err) {
      console.error("Error fetching countries:", err);
      setCountries([]);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchCountries();
  }, []);

  return (
    <CountryContext.Provider
      value={{
        country,
        setCountry,
        countries,
        loading,
        refreshCountries: fetchCountries,
      }}
    >
      {children}
    </CountryContext.Provider>
  );
};

// Custom hook for consuming context
export const useCountry = (): CountryContextType => {
  const context = useContext(CountryContext);
  if (!context) throw new Error("useCountry must be used within a CountryProvider");
  return context;
};
