import jwt, { type SignOptions, type Secret, type JwtPayload } from 'jsonwebtoken';

export async function generateJwtToken({
    payload,
    secret,
    expiresIn
}: {
    payload: object;
    secret: Secret;
    expiresIn: SignOptions['expiresIn'];
}): Promise<string> {
    try {
        const token = jwt.sign(payload, secret, { expiresIn });
        return token;
    } catch (error) {
        console.error('Error generating JWT token:', error);
        throw new Error('Failed to generate JWT token');
    }
}


export async function verifyJwtToken({
    token,
    secret
}: {
    token: string;
    secret: Secret;
}): Promise<JwtPayload | null> {
    try {
        const decoded = jwt.verify(token, secret) as JwtPayload;

        if (decoded.exp && Date.now() >= decoded.exp * 1000) {
            console.warn('Token has expired');
            return null;
        }

        return decoded;
    } catch (error: any) {
        if (error.name === 'TokenExpiredError') {
            console.warn('JWT verification failed: Token expired');
        } else if (error.name === 'JsonWebTokenError') {
            console.warn('JWT verification failed: Invalid token');
        } else {
            console.error('Unexpected JWT verification error:', error);
        }
        return null;
    }
}

