import { Prisma } from '@prisma/client';

export type SupportedLocale = 'en' | 'fr' | 'de' | 'it' | 'pt' | 'es';

export const CATEGORY_MULTILINGUAL_FIELDS = {
    name_en: true,
    name_fr: true,
    name_de: true,
    name_it: true,
    name_pt: true,
    name_es: true,
    description_en: true,
    description_fr: true,
    description_de: true,
    description_it: true,
    description_pt: true,
    description_es: true,
} as const;

export type MultilingualCategory = {
    name_en?: string | null;
    name_fr?: string | null;
    name_de?: string | null;
    name_it?: string | null;
    name_pt?: string | null;
    name_es?: string | null;
    description_en?: string | null;
    description_fr?: string | null;
    description_de?: string | null;
    description_it?: string | null;
    description_pt?: string | null;
    description_es?: string | null;
    [key: string]: any;
};

export function parseLocale(locale?: string | null): SupportedLocale {
    if (!locale) return 'en';
    const shortLocale = locale.split('-')[0].toLowerCase();
    const supportedLocales: SupportedLocale[] = ['en', 'fr', 'de', 'it', 'pt', 'es'];
    return supportedLocales.includes(shortLocale as SupportedLocale)
        ? (shortLocale as SupportedLocale)
        : 'en';
}

export function getCategoryName(
    category: MultilingualCategory | null | undefined,
    locale: SupportedLocale = 'en'
): string {
    if (!category) return '';
    const nameField = `name_${locale}` as keyof MultilingualCategory;
    const name = category[nameField] as string | null | undefined;
    return name || category.name_en || '';
}

export function getCategoryDescription(
    category: MultilingualCategory | null | undefined,
    locale: SupportedLocale = 'en'
): string {
    if (!category) return '';
    const descField = `description_${locale}` as keyof MultilingualCategory;
    const description = category[descField] as string | null | undefined;
    return description || category.description_en || '';
}
